/******************************************************************************
 * %Id: dev-share.c 117 2010-02-17 12:28:13Z 4011017 $
 *
 * FileName		:dev-share.c
 *
 * Description	:WAIT-QUEUE library for communication driver
 *
 *
 * Copyright	:Panasonic Co.
 *
 *****************************************************************************/
#define IOSC_NO_INCLUDE_SOURCE
#include "dev-share.h"

#include <linux/kernel.h> /* We run in the kernel so we need this */
#include <linux/poll.h>
#include <linux/iosc/iosc-dev.h>

#include "iprq.h"
#include "iprq_common.h"

/*================================================================
 * function of packet queue operation
 *================================================================*/

ssize_t read_datapacket( struct iosc_packet_queue_t *queue, char *buf, size_t nbytes )
{
    size_t copy_size   = 0;
    size_t read_bytes  = 0;
    size_t packet_size = 0;
    char *p;

    while( nbytes > 0 ){
        if (queue->remain == NULL && queue->offset == 0) {   /* new read */
            queue->remain = get_datapacket( queue );
            if( queue->remain == NULL ){   /* no data */
		    break;
            }
        }
        packet_size = queue->remain->length - queue->offset;
	p = buf + read_bytes;
        if( nbytes >= packet_size ){   /* request size >= packet size */
            copy_size = packet_size;
            if( copy_to_user( p, queue->remain->data+queue->offset, copy_size ) ){
                return -EFAULT;
            }
	    LFREE_MEM_ADDR(queue->remain);
            queue->offset = 0;
            queue->remain = NULL;
        }
        else{   /* request size < packet size */
            copy_size = nbytes;
            if( copy_to_user( p, queue->remain->data+queue->offset, copy_size ) ){
                return -EFAULT;
            }
            queue->offset += copy_size;
        }
        nbytes -= copy_size;
        read_bytes += copy_size;
    }
    return read_bytes;
}


ssize_t write_datapacket(unsigned int listid, char *buf, size_t nbytes)
{
    struct datapacket *packet;
    int memid;
    size_t copy_size  = 0;
    size_t write_size = 0;
    
    while( nbytes > 0 ){
        if( nbytes > IOSC_MAX_KMALLOC ){
            copy_size = IOSC_MAX_KMALLOC;
        }
        else{
            copy_size = nbytes;
        }
	memid = LALLOC_MEM(sizeof(struct datapacket) + sizeof(char) * copy_size);
        if (memid < 0) {
		break;
        }
	packet = (struct datapacket*)LATTACH_MEM(memid);
	if (copy_from_user(packet->data, buf, copy_size)) {
		LDETACH_MEM(packet);
		LFREE_MEM(memid);
		return -EFAULT;
	}
	LDETACH_MEM(packet);
        packet->length = copy_size;
        buf += copy_size;
        put_datapacket(listid, memid);
        nbytes -= copy_size;
        write_size += copy_size;
    }
    return write_size;
}

void wake_up_datapacket( struct iosc_packet_queue_t *queue )
{
    wake_up_interruptible( &queue->wait );
    return;
}

void wait_datapacket( struct iosc_packet_queue_t *queue )
{
    wait_event_interruptible( queue->wait, !list_empty(&queue->datapacket_list) );
    return;
}

int send_ctlpacket(unsigned int listid, unsigned int cmd, void *args, size_t size)
{
    struct ctlpacket *packet;
	int memid;

	memid = LALLOC_MEM( sizeof(struct ctlpacket) );
	if (memid < 0) {
		return -ENOMEM;
	}
	packet = (struct ctlpacket*)LATTACH_MEM(memid);

    packet->cmd = cmd;
	if (copy_from_user(&packet->args, args, sizeof(int))) {
		LDETACH_MEM(packet);
		LFREE_MEM(memid);
		return -EFAULT;
	}
    packet->length = size;

	LDETACH_MEM(packet);

	return put_ctlpacket(listid, memid);
}

int send_ctlpacket_self(struct iosc_packet_queue_t *queue, unsigned int cmd, void *args, size_t size)
{
    struct ctlpacket *packet;
	unsigned long flags;
    packet = (struct ctlpacket *)alloc_memory( sizeof(struct ctlpacket) );
    if( packet == NULL ){
        return -ENOMEM;
    }
    packet->cmd = cmd;
	if (copy_from_user(&packet->args, args, sizeof(int))) {
		free_memory(packet);
		return -EFAULT;
	}
    packet->length = size;

	spin_lock_irqsave(&queue->ctlpacket_lock, flags);
	list_add_tail((struct list_head *)packet, &queue->ctlpacket_list);
	wake_up_interruptible(&queue->wait);
	spin_unlock_irqrestore(&queue->ctlpacket_lock, flags);
    return 0;
}


int receive_ctlpacket( struct iosc_packet_queue_t *queue, struct iosc_read_packet_t *buf )
{
    struct ctlpacket *packet;

    wait_packet_queue( queue );
    packet = get_ctlpacket( queue );
    if( packet == NULL ){
        return -ENODATA;
    }
    /* response packet : kernel space -> memcpy */
    /* send packet : user space -> copy_to_user after return */
    memcpy( buf, &packet->cmd, sizeof(struct iosc_read_packet_t) );

    LFREE_MEM_ADDR(packet);
    return 0;
}

int init_datablock( struct iosc_data_block_t *block, int nblk )
{
    int i;

    for( i = 0; i < nblk; ++i ){
        init_waitqueue_head( &block[i].wait );
	spin_lock_init(&block[i].lock);
        block[i].length = 0;
        block[i].data = NULL;
    }

    return 0;
}

int clean_datablock(struct iosc_data_block_t *block, int nblk)
{
	int i;
	for (i = 0; i < nblk; ++i) {
		if (block[i].data) {
			LFREE_MEM_ADDR(block[i].data - sizeof(size_t));
			block[i].data = NULL;
			block[i].length = 0;
		}
	}
	return 0;
}

int get_datablock( struct iosc_data_block_t *block, char *data, size_t length )
{
#if 0
    if( block->length <= 0 ){
        wait_event_interruptible( block->wait, !( block->length <= 0 ) );
    }
#else
	unsigned long flags;
	DECLARE_WAITQUEUE(wait, current);
	set_current_state(TASK_INTERRUPTIBLE);
	add_wait_queue(&block->wait, &wait);
	spin_lock_irqsave(&block->lock, flags);
	while (block->length <= 0) {
		spin_unlock_irqrestore(&block->lock, flags);
		schedule();
		set_current_state(TASK_INTERRUPTIBLE);
		spin_lock_irqsave(&block->lock, flags);
	}
	set_current_state(TASK_RUNNING);
	remove_wait_queue(&block->wait, &wait);
#endif
    if( block->length > length ){
	spin_unlock_irqrestore(&block->lock, flags);
        return -ENOMEM;
    }
    if( copy_to_user( data, block->data, length ) ){
	spin_unlock_irqrestore(&block->lock, flags);
        return -EFAULT;
    }
    spin_unlock_irqrestore(&block->lock, flags);
    return 0;
}


int set_datablock(unsigned int listid, char *data, size_t length)
{
#if 0
    if( block->data != NULL ){
        free_memory( block->data );
        block->length = 0;
    }
    if( ( length == 0 ) || ( data == NULL ) ){
        block->data = NULL;
        block->length = 0;
        return 0;
    }

    block->data = alloc_memory( sizeof(char)*length );
    if( block->data == NULL ){
        return -ENOMEM;
    }
    if( copy_from_user( block->data, data, length ) ){
        return -EFAULT;
    }
    block->length = length;
    wake_up_interruptible( &block->wait );
#else
	void *shmem;
	int memid;
	if ((length == 0) || (data == NULL)) {
		memid = -1;
		LBLOCK_REQUEST(listid, memid);
		return 0;
	}
	memid = LALLOC_MEM(sizeof(char) * length + sizeof(size_t));
	if (memid < 0) {
		return -ENOMEM;
	}
	shmem = LATTACH_MEM(memid);
	if (copy_from_user(shmem + sizeof(size_t), data, length)) {
		return -EFAULT;
	}
	*(size_t *)shmem = length;
	LDETACH_MEM(shmem);
	LBLOCK_REQUEST(listid, memid);
	LWAKE_UP_INTERRUPTIBLE(listid);
#endif
    return 0;
}



/*================================================================
 * function of control packet operation
 *================================================================*/

/*
 * int put_ctlpacket(unsigned int listid, int memid)
 *-----------------------------------------------------------------------------
 * function: put control packet in wait queue of device, and release read_wait
 * argument: listid   receive queue information
 *           memid    send packet memory id
 * return  :  0			: success
 * comment :
 */
int put_ctlpacket(unsigned int listid, int memid) /* memid of a ctlpacket */
{
//    spin_lock( &to->ctlpacket_lock );
//    list_add_tail( (struct list_head *)packet, &to->ctlpacket_list );
//    wake_up_interruptible( &to->wait );
	LLIST_ADD_TAIL(memid, listid);
	LWAKE_UP_INTERRUPTIBLE(listid);
//    spin_unlock( &to->ctlpacket_lock );
    return 0;
}


/*
 * struct ctlpacket *get_ctlpacket( struct iosc_packet_queue_t *from )
 *-----------------------------------------------------------------------------
 * function: get control packet from wait queue of device.
 *           if No packet, then wait
 * argument: from      receive queue
 * return  :  except NULL         : success(packet address)
 *            NULL                : No packet
 * comment :
 */
struct ctlpacket *get_ctlpacket( struct iosc_packet_queue_t *from )
{
    struct ctlpacket *packet;
    unsigned long flags;
    spin_lock_irqsave(&from->ctlpacket_lock, flags);
    if (list_empty(&from->ctlpacket_list)) {
	spin_unlock_irqrestore(&from->ctlpacket_lock, flags);
        return NULL;
    }
    packet = (struct ctlpacket *)from->ctlpacket_list.next;
    list_del( (struct list_head *)packet );
    spin_unlock_irqrestore(&from->ctlpacket_lock, flags);
    return packet;
}


/*================================================================
 * function of data packet operation
 *================================================================*/

/*
 * int put_datapacket(unsigned int listid, int memid)
 *-----------------------------------------------------------------------------
 * function: put data packet in wait queue of device, and release read_wait
 * argument: listid   receive queue information
 *           memid    send packet memory id
 * return  :  0			: success
 * comment :
 */
int put_datapacket(unsigned int listid, int memid) /* memid of a datapacket */
{
//    spin_lock( &to->datapacket_lock );
//    list_add_tail( (struct list_head *)packet, &to->datapacket_list );
//    wake_up_interruptible( &to->wait );
	LLIST_ADD_TAIL(memid, listid);
	LWAKE_UP_INTERRUPTIBLE(listid);
//    spin_unlock( &to->datapacket_lock );
    return 0;
}

/*
 * struct datapacket *get_datapacket( struct iosc_packet_queue_t *from )
 *-----------------------------------------------------------------------------
 * function: get data packet from wait queue
 * argument: from      receive queue
 * return  :  except NULL         : success(data packet address)
 *            NULL                : No packet
 * comment :
 */
struct datapacket *get_datapacket( struct iosc_packet_queue_t *from )
{
    struct datapacket *packet;
	unsigned long flags;

    spin_lock_irqsave(&from->datapacket_lock, flags);
    if (list_empty(&from->datapacket_list)){
	spin_unlock_irqrestore(&from->datapacket_lock, flags);
        return NULL;
    }

    packet = (struct datapacket *)from->datapacket_list.next;
    list_del( (struct list_head *)packet );
    spin_unlock_irqrestore(&from->datapacket_lock, flags);

    return packet;

}

/*
 * void wait_packet_queue( struct iosc_packet_queue_t *queue )
 *-----------------------------------------------------------------------------
 * function: wait control packet
 * argument: queue      receive queue
 * return  : nothing
 * comment :
 */
void wait_packet_queue( struct iosc_packet_queue_t *queue )
{
    wait_event_interruptible( queue->wait, !list_empty(&queue->ctlpacket_list) );
}


/*
 * void init_packet_queue( struct iosc_packet_queue_t *queue )
 *-----------------------------------------------------------------------------
 * function: initialize packet queue
 * argument: queue      receive queue
 * return  : nothing
 * comment :
 */
void init_packet_queue( struct iosc_packet_queue_t *queue )
{
    init_waitqueue_head( &queue->wait );
    INIT_LIST_HEAD( &queue->ctlpacket_list );
    INIT_LIST_HEAD( &queue->datapacket_list );
    spin_lock_init( &queue->ctlpacket_lock );
    spin_lock_init( &queue->datapacket_lock );
    queue->offset = 0;
    queue->remain = NULL;
}

/*
 * clean_packet_queue( struct iosc_packet_queue_t *queue )
 *-----------------------------------------------------------------------------
 * function: clean control packet list
 * argument: queue      receive queue
 * return  : nothing
 * comment :
 */
void
clean_packet_queue( struct iosc_packet_queue_t *queue )
{
	struct list_head *p, *tmp;
	unsigned long flags;
	local_irq_save(flags);
	spin_lock(&queue->ctlpacket_lock);
	list_for_each_safe(p, tmp, &queue->ctlpacket_list) {
		list_del(p);
		LFREE_MEM_ADDR(p);
	}
	spin_unlock(&queue->ctlpacket_lock);
	spin_lock(&queue->datapacket_lock);
	list_for_each_safe(p, tmp, &queue->datapacket_list) {
		list_del(p);
		LFREE_MEM_ADDR(p);
	}
	spin_unlock(&queue->datapacket_lock);
	local_irq_restore(flags);
}

int malloc_count = 0;
int free_count = 0;
static DEFINE_MUTEX(count_mutex);

void *alloc_memory( size_t size )
{
    void *ptr = 0;
    int id;
    id = LALLOC_MEM(size);
    if (id < 0) {
	    printk("alloc_memory failed.(size=%d)\n", (int)size);
	    return ptr;
    }
    ptr = LATTACH_MEM(id);
    mutex_lock(&count_mutex);
    malloc_count += 1;
    mutex_unlock(&count_mutex);

    return ptr;
}


void free_memory( void *ptr )
{
	LDETACH_MEM(ptr);
	LFREE_MEM_ADDR(ptr);
    mutex_lock(&count_mutex);
    free_count += 1;
    mutex_unlock(&count_mutex);
}

void *local_alloc_memory(size_t size)
{
	return kmalloc(sizeof(char) * size, GFP_KERNEL);
}

void local_free_memory(void *ptr)
{
	kfree(ptr);
}

/* etc... module setting */
int __init iosc_share_init(void)
{
  printk(KERN_INFO "<dev-share> init\n");
  return 0;
}

void __exit iosc_share_exit(void)
{
  printk(KERN_INFO "<dev-share> exit\n");
}

#if 0   /*  if dynamic link, it is necessary */
//EXPORT_SYMBOL(list_count);
//EXPORT_SYMBOL(put_ctlpacket);
//EXPORT_SYMBOL(get_ctlpacket);
//EXPORT_SYMBOL(put_datapacket);
//EXPORT_SYMBOL(get_datapacket);
//EXPORT_SYMBOL(wait_packet_queue);
#endif

#ifdef MODULE_LICENSE
MODULE_LICENSE("Proprietary");
#endif
